/**
* \file: message_dispatcher.c
*
* \version: $Id:$
*
* \release: $Name:$
*
* \component: automounter
*
* \author: Marko Hoyer / ADIT / SWGII / mhoyer@de.adit-jv.com
*
* \copyright (c) 2010, 2011 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
*
***********************************************************************/
#include "app_iface/message_dispatcher.h"

#include "control/automounter.h"
#include "utils/logger.h"
#include "utils/automounter_types_internal.h"
#include "ipc/request_messages.h"
#include "ipc/info_messages.h"
#include "app_iface/message_sender.h"
#include "ipc/message_recvr.h"

#include "model/device_list.h"
#include "model/partition_list.h"

//------------------------------------- private attributes ------------------------------------------------------------
//---------------------------------------------------------------------------------------------------------------------

//------------------------------------- private member declaration ----------------------------------------------------

static void message_dispatcher_on_get_snapshot_msg(connection_ctx_t *ctx);
static void message_dispatcher_on_umount_request_msg(connection_ctx_t *ctx);
static void message_dispatcher_on_remount_request_msg(connection_ctx_t *ctx);
static void message_dispatcher_on_set_appinfo_msg(connection_ctx_t *ctx);

static app_iface_request_ctx_t *message_dispatcher_create_request_ctx(int request_id,void *callback_func,
		connection_ctx_t *ctx);

static void message_dispatcher_umount_part_callback(partition_t *part, partition_state_t previous_state, void *data);
static void message_dispatcher_umount_dev_callback(device_t *device, device_state_t previous_state, void *data);
static void message_dispatcher_remount_callback(partition_t *part, partition_state_t previous_state,void *data);

static error_code_t message_dispatcher_send_snapshot(connection_ctx_t *ctx,int request_id,snapshot_scope_t scope);

//---------------------------------------------------------------------------------------------------------------------

//------------------------------------- public member definition ------------------------------------------------------
void message_dispatcher_on_new_message(connection_ctx_t *ctx)
{
	message_type_t type;
	message_buffer_t *recvr_buffer;
	recvr_buffer=socket_server_get_receiver_buffer(ctx);

	type=message_recvr_get_msg_type(recvr_buffer);
	switch(type)
	{
		case SET_APPINFO_MSG:
			message_dispatcher_on_set_appinfo_msg(ctx);
			break;
		case GET_SNAPSHOT_MSG:
			message_dispatcher_on_get_snapshot_msg(ctx);
			break;
		case UMOUNT_REQUEST_MSG:
			message_dispatcher_on_umount_request_msg(ctx);
			break;
		case REMOUNT_REQUEST_MSG:
			message_dispatcher_on_remount_request_msg(ctx);
			break;
		case __NO_MSG:
		case SNAPSHOT_COMPLETE_MSG:
		case DEVICE_STATE_CHANGED_MSG:
		case PARTITION_STATE_CHANGED_MSG:
		case REQUEST_DONE_MSG:
		case __LAST_MSG_MARKER:
			logger_log_error("MESSAGE_DISPATCHER - Unknown message type: %d.",(int)type);
			break;
	}
}
//---------------------------------------------------------------------------------------------------------------------

//------------------------------------- private member definition -----------------------------------------------------
static void message_dispatcher_on_set_appinfo_msg(connection_ctx_t *ctx)
{
	const char *app_identifier;
	event_mask_t *event_mask_ptr;
	message_buffer_t *recvr_buffer;
	recvr_buffer=socket_server_get_receiver_buffer(ctx);

	if (request_message_extract_appinfo_msg(recvr_buffer,&app_identifier, &event_mask_ptr)!=RESULT_OK)
	{
		logger_log_error("MESSAGE_DISPATCHER - appinfo message invalid. Ignoring it!");
		return;
	}

	socket_server_set_connection_appinfo(ctx, app_identifier,*event_mask_ptr);
	logger_log_debug("MESSAGE_DISPATCHER - Application sent us an identifier('%s') and an event mask('0x%X').",
			app_identifier,(int)*event_mask_ptr);
}

static void message_dispatcher_on_get_snapshot_msg(connection_ctx_t *ctx)
{
	snapshot_request_message_t *msg;
	message_buffer_t *recvr_buffer;
	recvr_buffer=socket_server_get_receiver_buffer(ctx);

	if (request_message_extract_snapshot_request_msg(recvr_buffer, &msg)!=RESULT_OK)
	{
		logger_log_error("MESSAGE_DISPATCHER - snapshot request message invalid. Ignoring it!");
		return;
	}

	logger_log_debug("MESSAGE_DISPATCHER - snapshot request message received from application '%s':",
			socket_server_get_connection_appidentifier(ctx));
	logger_log_debug("scope: %d, request_id: %d",(int)msg->scope,(int)msg->request_id);

	if (message_dispatcher_send_snapshot(ctx,msg->request_id,msg->scope)==RESULT_OK)
		message_sender_send_snapshot_done_msg(ctx, msg->request_id);
}

static error_code_t message_dispatcher_send_snapshot(connection_ctx_t *ctx,int request_id,snapshot_scope_t scope)
{
	error_code_t result=RESULT_OK;
	device_t *device;
	device_list_iterator_t dev_list_itr;

	device=device_list_first_element(&dev_list_itr);
	while (device!=NULL && result==RESULT_OK)
	{
		device_info_t dev_info;
		partition_t *partition;
		partition_iterator_t part_list_itr;

		// we need the device for partition info update events. So it need to be extracted even if
		// the scope is only SNAPSHOT_PARTITIONS_ONLY or SNAP_SHOT_MOUNTEDED_PARTITIONS_ONLY
		//(believe me, I stumbled over this ;))
		app_iface_extract_dev_info(&dev_info,device);

		if (scope==SNAPSHOT_COMPLETE)
			result=message_sender_send_device_update_msg(ctx, &dev_info,request_id);

		partition=device_first_partition(device,&part_list_itr);
		while (partition!=NULL && result==RESULT_OK)
		{
			partition_info_t part_info;
			if (scope != SNAPSHOT_MOUNTED_PARTITIONS_ONLY ||
					partition_get_state(partition)==PARTITION_MOUNTED)
			{
				app_iface_extract_part_info(&part_info,partition);
				result=message_sender_send_partition_update_msg(ctx,&dev_info,&part_info,request_id);
			}
			partition=device_next_partition(&part_list_itr);
		}
		device=device_list_next_element(&dev_list_itr);
	}
	return result;
}

static void message_dispatcher_on_umount_request_msg(connection_ctx_t *ctx)
{
	error_code_t result;
	request_umount_message_t *msg;
	app_iface_request_ctx_t *request_ctx;
	message_buffer_t *recvr_buffer;
	recvr_buffer=socket_server_get_receiver_buffer(ctx);

	if (request_message_extract_umount_request_msg(recvr_buffer,&msg)!=RESULT_OK)
	{
		logger_log_error("MESSAGE_DISPATCHER - umount request message invalid. Ignoring it!");
		return;
	}

	request_ctx=message_dispatcher_create_request_ctx(msg->request_id,msg->callback_func, ctx);

	if (msg->id_type==ID_TYPE_MOUNTPOINT)
		result=app_iface_unmount_mountpoint_request(msg->identifier, message_dispatcher_umount_part_callback, request_ctx);
	else if (msg->id_type==ID_TYPE_PARTITION_ID)
		result=app_iface_unmount_partition_request(msg->identifier, message_dispatcher_umount_part_callback, request_ctx);
	else
		result=app_iface_unmount_device_request(msg->identifier, message_dispatcher_umount_dev_callback, request_ctx);

	if (result!=RESULT_OK)
	{
		message_sender_send_request_done_msg(request_ctx, result,"Umount failed.");
		free(request_ctx);
	}

	//in case of RESULT_OK, the request_ctx structure is freed in the callback
}

static void message_dispatcher_on_remount_request_msg(connection_ctx_t *ctx)
{
	error_code_t result;
	request_remount_message_t *msg;
	app_iface_request_ctx_t *request_ctx;
	message_buffer_t *recvr_buffer;
	recvr_buffer=socket_server_get_receiver_buffer(ctx);

	if (request_message_extract_remount_request_msg(recvr_buffer,&msg)!=RESULT_OK)
	{
		logger_log_error("MESSAGE_DISPATCHER - Remount request message invalid. Ignoring it!");
		return;
	}
	request_ctx=message_dispatcher_create_request_ctx(msg->request_id,msg->callback_func, ctx);

	if (msg->id_type==ID_TYPE_MOUNTPOINT)
		result=app_iface_remount_request_mp(msg->identifier,msg->options,
				message_dispatcher_remount_callback, request_ctx);
	else if (msg->id_type==ID_TYPE_PARTITION_ID)
		result=app_iface_remount_request_id(msg->identifier, msg->options,
				message_dispatcher_remount_callback, request_ctx);
	else
	{
		logger_log_error("MESSAGE_DISPATCHER - Got a remount request for a device. Who sends this?");
		message_sender_send_request_done_msg(request_ctx,RESULT_INVALID,"Remount request for a device does not make sense.");
		free(request_ctx);
		return;
	}

	if (result!=RESULT_OK)
	{
		message_sender_send_request_done_msg(request_ctx, result,"Remount failed while checking the conditions.");
		free(request_ctx);
	}

	//in case of RESULT_OK, the request_ctx structure is freed in the callback
}

static app_iface_request_ctx_t *message_dispatcher_create_request_ctx(int request_id,void *callback_func,
		connection_ctx_t *ctx)
{
	app_iface_request_ctx_t *request_ctx;
	request_ctx=malloc(sizeof(app_iface_request_ctx_t));
	if (request_ctx==NULL)
	{
		automounter_die_on_resource_issues();
		return NULL;
	}

	request_ctx->request_id=request_id;
	request_ctx->remote_callback_ptr=callback_func;
	request_ctx->ctx=ctx;

	return request_ctx;
}


static void message_dispatcher_umount_dev_callback(device_t *dev, device_state_t previous_state, void *data)
{
    device_state_t new_state;
    app_iface_request_ctx_t *request_ctx;

    if (previous_state!=DEVICE_UNMOUNTING)
    {
        logger_log_error("MESSAGE_DISPATCHER - We got a state change callback for a device that was"
                " not in state UNMOUNTING. Error in the implementation!!!");
        return;
    }

    request_ctx=(app_iface_request_ctx_t *)data;
    new_state=device_get_state(dev);
    if (new_state==DEVICE_UNMOUNTED)
        message_sender_send_request_done_msg(request_ctx,RESULT_OK,NULL);
    else if (new_state==DEVICE_AUTOMOUNTED)
    {
        message_sender_send_request_done_msg(request_ctx,RESULT_UMOUNT_ERR, "Unable to unmount one of the "
                "device's partitions.");
    }
    else
    {
        //nothing to do here
    }

    free(request_ctx);
}

static void message_dispatcher_umount_part_callback(partition_t *part, partition_state_t previous_state, void *data)
{
    partition_state_t new_state;
    error_code_t result;
    app_iface_request_ctx_t *request_ctx;

    if (previous_state!=PARTITION_UNMOUNTING)
    {
        logger_log_error("MESSAGE_DISPATCHER - We got a state change callback for a partition that was not in state UNMOUNTING."
                "Error in the implementation!!!");
        return;
    }

    request_ctx=(app_iface_request_ctx_t *)data;
    new_state=partition_get_state(part);
    if (new_state==PARTITION_UNMOUNTED)
        message_sender_send_request_done_msg(request_ctx,RESULT_OK,NULL);
    else if (new_state==PARTITION_MOUNTED)
    {
        result=partition_get_error(part);
        message_sender_send_request_done_msg(request_ctx,result,"Unable to unmount the partition.");
    }
    else
    {
        //nothing to do here
    }

    free(request_ctx);
}

static void message_dispatcher_remount_callback(partition_t *part, partition_state_t previous_state, void *data)
{
    error_code_t result;
    app_iface_request_ctx_t *request_ctx;

    if (previous_state!=PARTITION_REMOUNTING)
    {
        logger_log_error("MESSAGE_DISPATCHER - We got a state change callback for a partition that was not"
        		" in state REMOUNTING. Error in the implementation!!!");
        return;
    }

    request_ctx=(app_iface_request_ctx_t *)data;
    result=partition_get_error(part);
    if (result==RESULT_OK)
        message_sender_send_request_done_msg(request_ctx,RESULT_OK,NULL);
    else
        message_sender_send_request_done_msg(request_ctx,result, "Remount failed after trying to process it.");

    free(request_ctx);
}
//---------------------------------------------------------------------------------------------------------------------
